<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../helpers/helpers.php';
require_once '../helpers/permission_modal.php';
require_once '../includes/functions.php';

// Check if user has permission to view customers
// Check if user has any customer-related permissions
$customerPermissions = [
    'view_customers', 'manage_customers', 'view_sales', 'generate_reports'
];

if (!hasAnyPermission($customerPermissions)) {
    $_SESSION['error'] = "You don't have permission to access customer sales reports.";
    header('Location: customers.php');
    exit;
}

if (false) { // Disable the old check
    exit;
}

// Get store info
$storeInfo = [
    'store_name' => '',
    'store_address' => '',
    'store_phone' => '',
    'store_email' => ''
];
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_key IN ('store_name', 'store_address', 'store_phone', 'store_email')");
$stmt->execute();
foreach ($stmt->fetchAll(PDO::FETCH_KEY_PAIR) as $key => $value) {
    $storeInfo[$key] = $value;
}

// Get customer ID from URL
$customer_id = isset($_GET['id']) ? intval($_GET['id']) : 0;

if ($customer_id <= 0) {
    die('Invalid customer ID');
}

// Get customer details
$stmt = $pdo->prepare("SELECT name, phone, email, address FROM customers WHERE id = ?");
$stmt->execute([$customer_id]);
$customer = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$customer) {
    die('Customer not found');
}

// Get all sales for this customer with properly calculated credit payment details
$stmt = $pdo->prepare("
    SELECT 
        s.id as sale_id,
        s.created_at,
        s.total_amount,
        s.payment_method,
        s.payment_status,
        si.product_id,
        si.quantity,
        si.price,
        p.name as product_name,
        ct.amount as credit_amount,
        COALESCE(credit_payments.paid_amount, 0) as paid_amount,
        COALESCE(credit_payments.waived_amount, 0) as waived_amount
    FROM sales s
    JOIN sale_items si ON s.id = si.sale_id
    JOIN products p ON si.product_id = p.id
    LEFT JOIN credit_transactions ct ON s.id = ct.sale_id AND ct.customer_id = s.customer_id
    LEFT JOIN (
        SELECT 
            cp.credit_transaction_id,
            SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END) as paid_amount,
            SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END) as waived_amount
        FROM credit_payments cp
        GROUP BY cp.credit_transaction_id
    ) credit_payments ON ct.id = credit_payments.credit_transaction_id
    WHERE s.customer_id = ?
    ORDER BY s.created_at DESC, si.id ASC
");
$stmt->execute([$customer_id]);
$sales = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Calculate comprehensive totals matching credit.php logic
$total_sales = 0;
$total_items = 0;
$total_paid = 0;
$total_waived = 0;
$total_outstanding = 0;
$unique_transactions = [];

// Get credit totals matching credit.php logic (only outstanding balances)
$stmt = $pdo->prepare("
    WITH credit_balances AS (
        SELECT 
            ct.customer_id,
            ct.amount,
            COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) as paid,
            COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as waived,
            ct.amount - COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) - 
            COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as balance
        FROM credit_transactions ct
        LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
        WHERE ct.customer_id = ?
        GROUP BY ct.id, ct.customer_id, ct.amount
        HAVING balance > 0.001
    )
    SELECT 
        COALESCE(SUM(cb.amount), 0) as total_credit,
        COALESCE(SUM(cb.paid), 0) as total_paid,
        COALESCE(SUM(cb.waived), 0) as total_waived,
        COALESCE(SUM(cb.balance), 0) as total_balance
    FROM credit_balances cb
");
$stmt->execute([$customer_id]);
$credit_totals = $stmt->fetch();

foreach ($sales as $sale) {
    $subtotal = $sale['quantity'] * $sale['price'];
    $total_items += $sale['quantity'];
    
    // Calculate paid amount for display purposes
    $paid_amount = 0;
    if (isset($sale['paid_amount']) && $sale['paid_amount'] > 0) {
        $paid_amount = $sale['paid_amount'];
    } elseif (strtolower($sale['payment_status']) == 'paid' && strtolower($sale['payment_method']) != 'credit') {
        $paid_amount = $subtotal;
    }
    
    // Only add to total_sales if status is paid and amount was actually paid
    if (strtolower($sale['payment_status']) == 'paid' && $paid_amount > 0) {
        $total_sales += $paid_amount;
    }
    // Also include credit sales that have been fully paid
    elseif (strtolower($sale['payment_method']) == 'credit' && isset($sale['credit_amount']) && $sale['credit_amount'] > 0) {
        $total_credit_paid = ($sale['paid_amount'] ?? 0) + ($sale['waived_amount'] ?? 0);
        if ($total_credit_paid >= $sale['credit_amount']) {
            // Credit is fully paid, include in total sales
            $total_sales += $sale['credit_amount'];
        }
    }
    
    // Track unique transactions
    $unique_transactions[$sale['sale_id']] = true;
}

// Use credit.php values for financial summary
$total_paid = $credit_totals['total_paid'] ?? 0;
$total_waived = $credit_totals['total_waived'] ?? 0; 
$total_outstanding = $credit_totals['total_balance'] ?? 0;

// Calculate percentages
$payment_completion_rate = $total_sales > 0 ? (($total_paid + $total_waived) / $total_sales) * 100 : 0;
$waived_percentage = $total_sales > 0 ? ($total_waived / $total_sales) * 100 : 0;
$outstanding_percentage = $total_sales > 0 ? ($total_outstanding / $total_sales) * 100 : 0;

// Group sales by date
$grouped_sales = [];
foreach ($sales as $sale) {
    $date = date('Y-m-d', strtotime($sale['created_at']));
    if (!isset($grouped_sales[$date])) {
        $grouped_sales[$date] = [];
    }
    $grouped_sales[$date][] = $sale;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sales Report - <?php echo htmlspecialchars($customer['name']); ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body {
            font-family: 'Segoe UI', 'Arial', sans-serif;
            background: #f7f9fb;
        }
        @media print {
            @page {
                size: A4 portrait;
                margin: 10mm 7.5mm 15mm 7.5mm;
            }
            body {
                width: 195mm;
                min-height: 297mm;
                background: #fff !important;
                margin: 0 auto;
                padding: 0;
            }
            .main-container {
                margin: 0 auto;
                padding: 0;
                width: 195mm;
                max-width: 195mm;
            }
            .centered-header {
                width: 195mm;
                max-width: 195mm;
                margin: 0 auto 10px auto;
                padding: 0;
                text-align: center;
            }
            .customer-details {
                width: 195mm;
                max-width: 195mm;
                margin: 0 auto 15px auto;
                padding: 10px 15px;
            }
            .table-responsive {
                width: 195mm !important;
                max-width: 195mm !important;
                margin: 0 auto;
                padding: 0;
                overflow-x: visible;
            }
            .sale-date {
                margin: 0 auto;
                padding: 0;
            }
            table {
                width: 195mm !important;
                max-width: 195mm !important;
                margin: 0 auto;
                padding: 0;
                font-size: 10px;
                table-layout: auto;
                border-collapse: collapse;
            }
            th {
                white-space: nowrap !important;
                font-weight: bold;
            }
            th:nth-child(1) { min-width: 16mm; }
            th:nth-child(2) { min-width: 25mm; }
            th:nth-child(3) { min-width: 12mm; }
            th:nth-child(4) { min-width: 14mm; }
            th:nth-child(5) { min-width: 18mm; }
            th:nth-child(6) { min-width: 16mm; }
            th:nth-child(7) { min-width: 16mm; }
            th:nth-child(8) { min-width: 16mm; }
            th:nth-child(9) { min-width: 18mm; }
            th:nth-child(10) { min-width: 14mm; }
            th:nth-child(11) { min-width: 22mm; }
            thead tr {
                background: #e5e7eb !important;
            }
            th, td {
                padding: 3px 4px !important;
                word-break: break-word;
                overflow-wrap: break-word;
                text-align: left;
                border: 1px solid #bbb !important;
            }
            tbody tr:nth-child(even) {
                background: #f4f6fa !important;
            }
            tbody tr:nth-child(odd) {
                background: #fff !important;
            }
            .credit-balance {
                color: #16a34a !important;
                font-weight: 600 !important;
            }
            .print-footer {
                position: fixed;
                bottom: 0;
                left: 0;
                width: 100%;
                text-align: right;
                font-size: 12px;
                color: #888;
                padding-right: 10mm;
                margin: 0;
                z-index: 1000;
            }
            .no-print {
                display: none !important;
            }
            .page-break {
                page-break-before: always;
            }
            .watermark {
                position: fixed;
                top: 50%;
                left: 50%;
                transform: translate(-50%, -50%);
                opacity: 0.08;
                font-size: 25px;
                color: #000;
                width: 100%;
                text-align: center;
                pointer-events: none;
                z-index: 0;
                white-space: pre-line;
                display: flex;
                flex-direction: column;
                align-items: center;
                justify-content: center;
            }
            .watermark-logo {
                max-width: 300px;
                max-height: 200px;
                margin-bottom: 20px;
                opacity: 0.9;
                filter: grayscale(100%);
            }
            .watermark-text {
                font-size: 25px;
                color: #000;
                opacity: 1;
                text-align: center;
                white-space: pre-line;
                line-height: 1;
                font-weight: bold;
            }
            .totals {
                font-size: 11px !important;
                margin-top: 5mm !important;
                padding-top: 5mm !important;
                page-break-inside: avoid;
                width: 195mm !important;
                max-width: 195mm !important;
                margin-left: auto !important;
                margin-right: auto !important;
            }
            .summary-row {
                display: flex !important;
                width: 100% !important;
                gap: 10px !important;
            }
            .summary-column {
                flex: 1 !important;
                width: 50% !important;
                border: 1px solid #ddd !important;
                padding: 8px !important;
                background: #f9f9f9 !important;
            }
            .summary-column h5 {
                font-size: 11px !important;
                margin-bottom: 8px !important;
                padding-bottom: 4px !important;
                border-bottom: 1px solid #ccc !important;
            }
            .summary-grid {
                font-size: 10px !important;
            }
            .summary-grid > div {
                padding: 2px 0 !important;
            }
        }
        @media screen {
            .watermark { display: none; }
            .print-footer { display: none; }
            .modern-btns {
                display: flex;
                justify-content: center;
                gap: 16px;
                margin: 0 0 30px 0;
                padding: 20px 0;
            }
            .modern-btns .btn {
                font-size: 1rem;
                padding: 10px 24px;
                border-radius: 24px;
                box-shadow: 0 2px 8px rgba(0,0,0,0.06);
                border: none;
                transition: background 0.2s, color 0.2s;
            }
            .modern-btns .btn-primary {
                background: #2563eb;
                color: #fff;
            }
            .modern-btns .btn-primary:hover {
                background: #1e40af;
            }
            .modern-btns .btn-secondary {
                background: #f3f4f6;
                color: #222;
            }
            .modern-btns .btn-secondary:hover {
                background: #e5e7eb;
            }
            .centered-header {
                background: #fff;
                border-radius: 16px;
                box-shadow: 0 2px 12px rgba(0,0,0,0.07);
                padding: 24px 0 16px 0;
                margin: 0 auto 18px auto;
                width: 100%;
                max-width: 100%;
                text-align: center;
            }
            .store-logo {
                margin-bottom: 10px;
            }
            .report-header {
                margin-bottom: 24px;
                font-weight: 600;
                font-size: 1.5rem;
                color: #222;
            }
            .customer-details {
                background: #fff;
                border-radius: 12px;
                box-shadow: 0 1px 6px rgba(0,0,0,0.05);
                padding: 18px 20px 10px 20px;
                margin: 0 auto 18px auto;
                width: 100%;
                max-width: 100%;
            }
            .sale-date {
                background-color: #e0e7ef;
                padding: 10px;
                margin-top: 20px;
                margin-bottom: 0px;
                border-radius: 8px;
                font-weight: 500;
                width: 100%;
            }
            .totals {
                margin-top: 0px;
                border-top: 2px solid #dee2e6;
                padding-top: 20px;
                background: #fff;
                border-radius: 0 0 10px 10px;
                box-shadow: 0 1px 6px rgba(0,0,0,0.04);
                width: 100%;
                max-width: 100%;
                margin-left: auto;
                margin-right: auto;
            }
            .summary-row {
                display: flex;
                width: 100%;
                gap: 20px;
                margin-bottom: 0;
            }
            .summary-column {
                flex: 1;
                width: 50%;
                border: 1px solid #e5e7eb;
                border-radius: 8px;
                padding: 16px;
                background: #f8fafc;
                box-shadow: 0 1px 3px rgba(0,0,0,0.1);
            }
            .summary-column h5 {
                font-size: 1.1rem;
                margin-bottom: 12px;
                padding-bottom: 8px;
                border-bottom: 2px solid #e5e7eb;
                color: #374151;
                font-weight: 600;
            }
            .summary-grid {
                font-size: 0.95rem;
            }
            .summary-grid > div {
                padding: 4px 0;
                border-bottom: 1px solid #f1f5f9;
            }
            .summary-grid > div:nth-last-child(-n+2) {
                border-bottom: none;
            }
            .table-responsive {
                background: #fff;
                border-radius: 10px 10px 0 0;
                box-shadow: 0 1px 6px rgba(0,0,0,0.04);
                margin-bottom: 0px;
            }
            table {
                font-size: 0.95rem;
                table-layout: auto;
            }
            thead tr {
                background: #f1f5f9;
            }
            th, td {
                padding: 7px 8px;
                word-break: break-word;
                overflow-wrap: break-word;
                text-align: left;
                white-space: nowrap !important;
            }
            tbody tr:nth-child(even) {
                background: #f9fafb;
            }
            tbody tr:hover {
                background: #e0e7ef;
            }
            .credit-balance {
                color: #16a34a;
                font-weight: 600;
            }
            body {
                padding: 20px;
            }
            .main-container {
                width: 100%;
                max-width: 180mm;
                margin: 0 auto;
            }
            .table-responsive {
                width: 100%;
                max-width: 100%;
                margin: 0 auto;
                overflow-x: visible;
            }
            th:nth-child(1) { min-width: 50px; }
            th:nth-child(2) { min-width: 110px; }
            th:nth-child(3) { min-width: 55px; }
            th:nth-child(4) { min-width: 65px; }
            th:nth-child(5) { min-width: 80px; }
            th:nth-child(6) { min-width: 65px; }
            th:nth-child(7) { min-width: 65px; }
            th:nth-child(8) { min-width: 70px; }
            th:nth-child(9) { min-width: 100px; }
            th:nth-child(10) { min-width: 65px; }
            th:nth-child(11) { min-width: 105px; }
        }
    </style>
</head>
<body>
    <div class="main-container">
        <div class="modern-btns no-print" style="margin-bottom: 0;">
            <button class="btn btn-secondary" onclick="window.location.href='customers.php'">
                &larr; Back
            </button>
            <button class="btn btn-primary" onclick="window.print()">
                Print
            </button>
        </div>
        <div class="watermark">
            <div class="watermark-logo">
                <?php echo getStoreLogo('watermark-logo', 'max-width:300px;max-height:200px;'); ?>
            </div>
            <div style="height: 20px;"></div>
            <div class="watermark-text">
                <?php
                $watermark = $storeInfo['store_name'];
                if (!empty($storeInfo['store_address'])) $watermark .= "\n" . $storeInfo['store_address'];
                if (!empty($storeInfo['store_phone'])) $watermark .= "\nPhone: " . $storeInfo['store_phone'];
                if (!empty($storeInfo['store_email'])) $watermark .= "\nEmail: " . $storeInfo['store_email'];
                echo nl2br(htmlspecialchars($watermark));
                ?>
            </div>
        </div>
    <div class="centered-header">
        <?php echo getStoreLogo('store-logo mb-2', 'max-width:210px;max-height:140px;'); ?>
        <h3 class="mb-0"><?php echo htmlspecialchars($storeInfo['store_name']); ?></h3>
        <div class="small text-muted">
            <?php if (!empty($storeInfo['store_address'])): ?>
                <div><?php echo htmlspecialchars($storeInfo['store_address']); ?></div>
            <?php endif; ?>
            <?php if (!empty($storeInfo['store_phone'])): ?>
                <div>Phone: <?php echo htmlspecialchars($storeInfo['store_phone']); ?></div>
            <?php endif; ?>
            <?php if (!empty($storeInfo['store_email'])): ?>
                <div>Email: <?php echo htmlspecialchars($storeInfo['store_email']); ?></div>
            <?php endif; ?>
        </div>
    </div>
    <hr class="my-3">

    <div class="report-header">
        <h2>Customer Sales Report</h2>
    </div>
    <hr class="my-3">

    <div class="customer-details">
        <h4><?php echo htmlspecialchars($customer['name']); ?></h4>
        <p>
            <strong>Phone:</strong> <?php echo htmlspecialchars($customer['phone']); ?><br>
            <strong>Email:</strong> <?php echo htmlspecialchars($customer['email']); ?><br>
            <strong>Address:</strong> <?php echo htmlspecialchars($customer['address']); ?>
        </p>
    </div>
    <hr class="my-3">

    <?php if (empty($sales)): ?>
        <div class="alert alert-info">No sales records found for this customer.</div>
    <?php else: ?>
        <?php foreach ($grouped_sales as $date => $day_sales): ?>
            <div class="sale-date">
                <h5><?php echo date('F d, Y', strtotime($date)); ?></h5>
            </div>
            <div class="table-responsive">
                <table class="table table-bordered table-striped">
                    <thead>
                        <tr>
                            <th>Sale ID</th>
                            <th>Product</th>
                            <th>Quantity</th>
                            <th>Price</th>
                            <th>Subtotal</th>
                            <th>Paid</th>
                            <th>Waived</th>
                            <th>Balance</th>
                            <th>Pymt Method</th>
                            <th>Status</th>
                            <th>Date/Time</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($day_sales as $sale): ?>
                            <tr>
                                <td><?php echo $sale['sale_id']; ?></td>
                                <td><?php echo htmlspecialchars($sale['product_name']); ?></td>
                                <td><?php echo $sale['quantity']; ?></td>
                                <td><?php echo number_format($sale['price'], 2); ?></td>
                                <td><?php echo number_format($sale['quantity'] * $sale['price'], 2); ?></td>
                                <td>
                                    <?php
                                    // Display paid amount if available, else fallback to subtotal if fully paid, else 0
                                    $subtotal = $sale['quantity'] * $sale['price'];
                                    if (isset($sale['paid_amount']) && $sale['paid_amount'] > 0) {
                                        $paid = $sale['paid_amount'];
                                        echo number_format($paid, 2);
                                    } elseif (strtolower($sale['payment_status']) == 'paid' && strtolower($sale['payment_method']) != 'credit') {
                                        $paid = $subtotal;
                                        echo number_format($paid, 2);
                                    } else {
                                        $paid = 0;
                                        echo '0.00';
                                    }
                                    ?>
                                </td>
                                <td>
                                    <?php
                                    // Display waived amount
                                    $waived = isset($sale['waived_amount']) ? $sale['waived_amount'] : 0;
                                    echo number_format($waived, 2);
                                    ?>
                                </td>
                                <td>
                                    <?php
                                    // Calculate balance
                                    $balance = $subtotal - $paid - $waived;
                                    if ($balance < 0) {
                                        echo '<span class="credit-balance">Credit: ' . number_format(abs($balance), 2) . '</span>';
                                    } else {
                                        echo number_format($balance, 2);
                                    }
                                    ?>
                                </td>
                                <td><?php echo ucfirst($sale['payment_method']); ?></td>
                                <td>
                                    <span class="badge <?php echo $sale['payment_status'] == 'paid' ? 'bg-success' : 'bg-warning'; ?>">
                                        <?php echo ucfirst($sale['payment_status']); ?>
                                    </span>
                                </td>
                                <td><?php echo date('Y-m-d H:i', strtotime($sale['created_at'])); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endforeach; ?>

        <div class="totals">
            <div class="summary-row">
                <div class="summary-column">
                    <h5><i class="fas fa-chart-line"></i> Financial Summary</h5>
                    <div class="summary-grid" style="display: grid; grid-template-columns: 1fr 1fr; gap: 8px; margin-bottom: 15px;">
                        <div><strong>Total Sales:</strong></div>
                        <div style="text-align: right; color: #2563eb; font-weight: 600;">Ksh <?php echo number_format($total_sales, 2); ?></div>
                        
                        <div><strong>Total Paid:</strong></div>
                        <div style="text-align: right; color: #16a34a; font-weight: 600;">Ksh <?php echo number_format($total_paid, 2); ?></div>
                        
                        <div><strong>Total Waived:</strong></div>
                        <div style="text-align: right; color: #ea580c; font-weight: 600;">Ksh <?php echo number_format($total_waived, 2); ?></div>
                        
                        <div><strong>Outstanding Balance:</strong></div>
                        <div style="text-align: right; color: #dc2626; font-weight: 600;">Ksh <?php echo number_format($total_outstanding, 2); ?></div>
                    </div>
                    
                    <div style="border-top: 1px solid #e5e7eb; padding-top: 10px; margin-top: 10px;">
                        <div style="font-size: 0.9em; color: #6b7280;">
                            <div style="margin-bottom: 5px;">
                                <strong>Payment Completion:</strong> 
                                <span style="color: <?php echo $payment_completion_rate >= 80 ? '#16a34a' : ($payment_completion_rate >= 50 ? '#ea580c' : '#dc2626'); ?>; font-weight: 600;">
                                    <?php echo number_format($payment_completion_rate, 1); ?>%
                                </span>
                            </div>
                            <?php if ($total_waived > 0): ?>
                            <div style="margin-bottom: 5px;">
                                <strong>Waived Amount:</strong> 
                                <span style="color: #ea580c; font-weight: 600;">
                                    <?php echo number_format($waived_percentage, 1); ?>%
                                </span>
                            </div>
                            <?php endif; ?>
                            <?php if ($total_outstanding > 0): ?>
                            <div>
                                <strong>Outstanding:</strong> 
                                <span style="color: #dc2626; font-weight: 600;">
                                    <?php echo number_format($outstanding_percentage, 1); ?>%
                                </span>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                <div class="summary-column">
                    <h5><i class="fas fa-shopping-cart"></i> Transaction Summary</h5>
                    <div class="summary-grid" style="display: grid; grid-template-columns: 1fr 1fr; gap: 8px; margin-bottom: 15px;">
                        <div><strong>Total Items:</strong></div>
                        <div style="text-align: right; font-weight: 600;"><?php echo number_format($total_items); ?></div>
                        
                        <div><strong>Unique Transactions:</strong></div>
                        <div style="text-align: right; font-weight: 600;"><?php echo count($unique_transactions); ?></div>
                        
                        <div><strong>Total Records:</strong></div>
                        <div style="text-align: right; font-weight: 600;"><?php echo count($sales); ?></div>
                        
                        <div><strong>Avg. Items/Transaction:</strong></div>
                        <div style="text-align: right; font-weight: 600;">
                            <?php echo count($unique_transactions) > 0 ? number_format($total_items / count($unique_transactions), 1) : '0.0'; ?>
                        </div>
                    </div>
                    
                    <div style="border-top: 1px solid #e5e7eb; padding-top: 10px; margin-top: 10px;">
                        <div style="font-size: 0.9em; color: #6b7280;">
                            <div style="margin-bottom: 5px;">
                                <strong>Avg. Transaction Value:</strong> 
                                <span style="font-weight: 600;">
                                    Ksh <?php echo count($unique_transactions) > 0 ? number_format($total_sales / count($unique_transactions), 2) : '0.00'; ?>
                                </span>
                            </div>
                            <div style="margin-bottom: 5px;">
                                <strong>Report Period:</strong> 
                                <span style="font-weight: 600;">
                                    <?php 
                                    if (!empty($sales)) {
                                        $dates = array_map(function($sale) { return strtotime($sale['created_at']); }, $sales);
                                        $min_date = min($dates);
                                        $max_date = max($dates);
                                        if (date('Y-m-d', $min_date) == date('Y-m-d', $max_date)) {
                                            echo date('M d, Y', $min_date);
                                        } else {
                                            echo date('M d, Y', $min_date) . ' - ' . date('M d, Y', $max_date);
                                        }
                                    } else {
                                        echo 'No data';
                                    }
                                    ?>
                                </span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    <?php endif; ?>

        <div class="print-footer">
            Generated on <?php echo date('F d, Y H:i:s'); ?>
            <br>
            <span style="font-size:11px;">Designed and developed by <a href="https://vicotechnologies.co.ke" target="_blank" style="color:#2563eb;text-decoration:none;">Vico Technologies</a> (0702214793)</span>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://kit.fontawesome.com/your-font-awesome-kit.js"></script>
</body>
</html>
<?php include '../includes/footer.php'; ?> 